/**********************************************************
 * Copyright (c) 2003-2024 Ijinus - All Rights Reserved.
 * This source code file is proprietary (closed source).
 **********************************************************
 * File     : codec.js
 * Version	: 1.1.7
 * Author	: Baptiste LE BOURHIS
 * Contact	: si@ijinus.fr
 * Site		: https://www.ijinus.com
 **********************************************************/

const LANG = "FR"; // fallback to EN if lang translation does not exists
const DEBUG = false;

const DATA_TYPES = Object.freeze({
	0: {
		label: 'Boolean',
		labelFR: 'Booléen',
		unit: '',
		isSigned: false,
		convertFn: (x) => Boolean(x),
	},
	1: {
		label: 'Unicode char',
		labelFR: 'Caractère unicode',
		unit: '',
		isSigned: false,
		convertFn: (x) => String.fromCharCode(x),
	},
	2: {
		label: 'Integer (unsigned)',
		labelFR: 'Entier (non signé)',
		unit: '',
		isSigned: false,
		convertFn: (x) => x,
	},
	3: {
		label: 'Integer (signed)',
		labelFR: 'Entier (signé)',
		unit: '',
		isSigned: true,
		convertFn: (x) => x,
	},
	4: {
		label: 'Real',
		labelFR: 'Réel',
		unit: '',
		isSigned: true,
		convertFn: (x) => ToFloat(x),
	},
	5: {
		label: 'Battery voltage (1/10 V)',
		labelFR: 'Tension batterie (1/10 V)',
		unit: 'V',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.1),
	},
	6: {
		label: 'Battery voltage (1/20 V)',
		labelFR: 'Tension batterie (1/20 V)',
		unit: 'V',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.05),
	},
	7: {
		label: 'Voltage signal',
		labelFR: 'Tension',
		unit: 'V',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.01),
	},
	8: {
		label: 'Current (1/10 mA)',
		labelFR: 'Courant (1/10 mA)',
		unit: 'mA',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.1),
	},
	9: {
		label: 'Current (1/20 mA)',
		labelFR: 'Courant (1/20 mA)',
		unit: 'mA',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.05),
	},
	10: {
		label: 'Current (1/100 mA)',
		labelFR: 'Courant (1/100 mA)',
		unit: 'mA',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.01),
	},
	11: {
		label: 'System temperature',
		labelFR: 'Température système',
		unit: '°C',
		isSigned: true,
		convertFn: (x) => x,
	},
	12: {
		label: 'Temperature (1/10 °C)',
		labelFR: 'Température (1/10 °C)',
		unit: '°C',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 0.1),
	},
	13: {
		label: 'Temperature (1/16 °C)',
		labelFR: 'Température (1/16 °C)',
		unit: '°C',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 0.0625),
	},
	14: {
		label: 'Height (unsigned)',
		labelFR: 'Hauteur (non signée)',
		unit: 'mm',
		isSigned: false,
		convertFn: (x) => x,
	},
	15: {
		label: 'Height (signed)',
		labelFR: 'Hauteur (signée)',
		unit: 'mm',
		isSigned: true,
		convertFn: (x) => x,
	},
	16: {
		label: 'Memory usage',
		labelFR: 'Utilisation mémoire',
		unit: '%',
		isSigned: false,
		convertFn: (x) => x,
	},
	17: {
		label: 'GSM signal strength',
		labelFR: 'Puissance signal GSM',
		unit: 'dBm',
		isSigned: false,
		convertFn: (x) => x - 120,
	},
	18: {
		label: 'RF signal strength',
		labelFR: 'Puissance signal RF',
		unit: 'dBm',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.5) - 140.0,
	},
	19: {
		label: 'Date (Unix Timestamp)',
		labelFR: 'Date (Unix Timestamp)',
		unit: 's',
		isSigned: false,
		convertFn: (x) => x,
	},
	20: {
		label: 'Overflow status',
		labelFR: 'État de surverse',
		unit: '',
		isSigned: false,
		convertFn: (x) => Boolean(x),
	},
	21: {
		label: 'Modbus register',
		labelFR: 'Registres Modbus',
		unit: '',
		isSigned: false,
		convertFn: (x) => x,
	},
	22: {
		label: 'Counter',
		labelFR: 'Compteur',
		unit: '',
		isSigned: false,
		convertFn: (x) => x,
	},
	23: {
		label: 'Capacitive saturation',
		labelFR: 'Saturation Capacitive',
		unit: '%',
		isSigned: false,
		convertFn: (x) => x,
	},
	24: {
		label: 'Velocity',
		labelFR: 'Vitesse',
		unit: 'mm/s',
		isSigned: true,
		convertFn: (x) => x,
	},
	25: {
		label: 'Quality of a measurement',
		labelFR: 'Qualité d\'une prise de mesure',
		unit: '%',
		isSigned: false,
		convertFn: (x) => x,
	},
	26: {
		label: 'Conductivity',
		labelFR: 'Conductivité',
		unit: 'μS/cm',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.01),
	},
	27: {
		label: 'Salinity',
		labelFR: 'Salinité',
		unit: 'g/Kg',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.01),
	},
	28: {
		label: 'Oxygen saturation',
		labelFR: 'Saturation en oxygène',
		unit: '%Sat',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.01),
	},
	29: {
		label: 'Dissolved oxygen',
		labelFR: 'Oxygène dissous',
		unit: 'mg/L',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.01),
	},
	30: {
		label: 'pH',
		labelFR: 'pH',
		unit: 'pH',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.01),
	},
	31: {
		label: 'Redox',
		labelFR: 'Redox',
		unit: 'mV',
		isSigned: true,
		convertFn: (x) => x,
	},
	32: {
		label: 'FNU Turbidity',
		labelFR: 'Turbidité Néphélo',
		unit: 'FNU',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.1),
	},
	33: {
		label: 'TU Turbidity',
		labelFR: 'Turbidité TU',
		unit: 'mg/L',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.1),
	},
	34: {
		label: 'Flow',
		labelFR: 'Débit',
		unit: 'm³/s',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 1.0e-6),
	},
	35: {
		label: 'Precipitation',
		labelFR: 'Précipitation',
		unit: 'mm',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.1),
	},
	36: {
		label: 'Distance',
		labelFR: 'Distance',
		unit: 'mm',
		isSigned: false,
		convertFn: (x) => x,
	},
	37: {
		label: 'Pressure',
		labelFR: 'Pression',
		unit: 'bar',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 0.01),
	},
	38: {
		label: 'Concentration',
		labelFR: 'Concentration',
		unit: 'ppm',
		isSigned: false,
		convertFn: (x) => x,
	},
	39: {
		label: 'Volume',
		labelFR: 'Volume',
		unit: 'm³',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 0.001),
	},
	40: {
		label: 'Duration',
		labelFR: 'Durée',
		unit: 's',
		isSigned: false,
		convertFn: (x) => x,
	},
	41: {
		label: 'Humidity',
		labelFR: 'Humidité',
		unit: '%RH',
		isSigned: false,
		convertFn: (x) => x,
	},
	42: {
		label: 'Weight',
		labelFR: 'Poids',
		unit: 't',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.001),
	},
	43: {
		label: 'Wattage',
		labelFR: 'Puissance',
		unit: 'W',
		isSigned: false,
		convertFn: (x) => x,
	},
	44: {
		label: 'Angle',
		labelFR: 'Angle',
		unit: '°',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 0.1),
	},
	45: {
		label: 'Concentration (1/10)',
		labelFR: 'Concentration (1/10)',
		unit: 'ppm',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 0.1),
	},
	46: {
		label: 'Duration',
		labelFR: 'Durée',
		unit: 'd',
		isSigned: false,
		convertFn: (x) => x,
	},
	47: {
		label: 'Height',
		labelFR: 'Hauteur',
		unit: 'm',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 0.001),
	},
	48: {
		label: 'Flow',
		labelFR: 'Débit',
		unit: 'm3/h',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 0.001),
	},
	49: {
		label: 'Volume',
		labelFR: 'Volume',
		unit: 'L',
		isSigned: true,
		convertFn: (x) => x,
	},
	50: {
		label: 'Energy',
		labelFR: 'Énergie',
		unit: 'mAh',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.001),
	},
	51: {
		label: 'Velocity',
		labelFR: 'Vitesse',
		unit: 'm/s',
		isSigned: true,
		convertFn: (x) => RoundFloat32(x * 0.001),
	},
	52: {
		label: 'Concentration',
		labelFR: 'Concentration',
		unit: 'μg/L',
		isSigned: false,
		convertFn: (x) => x,
	},
	53: {
		label: 'Percentage',
		labelFR: 'Pourcentage',
		unit: '%',
		isSigned: false,
		convertFn: (x) => RoundFloat32(x * 0.01),
	},
	54: {
		label: 'Quality',
		labelFR: 'Qualité',
		unit: '',
		isSigned: true,
		convertFn: (x) => x,
	},
	55: {
		label: 'Longitude',
		labelFR: 'Longitude',
		unit: '°',
		isSigned: true,
		convertFn: (x) => ToFloat(x),
	},
	56: {
		label: 'Latitude',
		labelFR: 'Latitude',
		unit: '°',
		isSigned: true,
		convertFn: (x) => ToFloat(x),
	}
});
const BIT_SIZE_TABLE = Object.freeze({
	0: 1,
	1: 4,
	2: 5,
	3: 6,
	4: 7,
	5: 8,
	6: 9,
	7: 10,
	8: 12,
	9: 14,
	10: 16,
	11: 20,
	12: 24,
	13: 28,
	14: 32
});
const SECONDS_INTERVAL_TABLE = Object.freeze({
	1: 1,
	2: 2,
	3: 3,
	4: 4,
	5: 5,
	6: 6,
	7: 10,
	8: 12,
	9: 15,
	10: 20,
	11: 30,
	12: 60,
	13: 2 * 60,
	14: 3 * 60,
	15: 4 * 60,
	16: 5 * 60,
	17: 6 * 60,
	18: 10 * 60,
	19: 12 * 60,
	20: 15 * 60,
	21: 20 * 60,
	22: 30 * 60,
	23: 60 * 60,
	24: 2 * 60 * 60,
	25: 3 * 60 * 60,
	26: 4 * 60 * 60,
	27: 6 * 60 * 60,
	28: 8 * 60 * 60,
	29: 12 * 60 * 60,
	30: 24 * 60 * 60,
	31: 0
});
const ASYNC_INTERVAL_TABLE = Object.freeze({
	0: {
		size: 6,
		convertFn: (x) => x
	},
	1: {
		size: 10,
		convertFn: (x) => (64 + x)
	},
	2: {
		size: 14,
		convertFn: (x) => (1088 + x)
	},
	3: {
		size: 18,
		convertFn: (x) => (17472 + x)
	}
});
const MIN_ASYNC_SIZE = 6;

function ToFloat(intValue, precision = 6) {
	const floatArray = new Float32Array(1);
    const intArray = new Uint32Array(floatArray.buffer);
	intArray[0] = intValue;
	return parseFloat(floatArray[0].toFixed(precision));
};

function RoundFloat32(floatValue) {
	return parseFloat(floatValue.toPrecision(7));
};

function DebugLog (msg) {
	if (DEBUG) console.log("[DEBUG] " + msg);
};

function GetBitsIterator(byteArray) {
	if (byteArray == null || byteArray.length < 1) return null;
	const _iterator = {
		byteArray: byteArray,
		byteIndex: 0, // Byte index in byte array
		bitIndex: 0, // Bit index inside the current byte
		remainingBits: byteArray.length * 8,
		offset: 8, // Bits offset, default to 1 byte
	};
	const iterator = {
		current: null,
		get length() {
			return _iterator.byteArray.length * 8
		},
		GetNext: (bitsPerElement = null, isSigned = false) => {
			let currentBitsPerElement = null;
			if (bitsPerElement != null) {
				// Using given offset only for this call of GetNext()
				currentBitsPerElement = _iterator.offset;
				iterator.SetBitsPerElement(bitsPerElement); // Using SetBitsPerElement to check given bitsPerElement format
			}

			let value = 0;
			let bitsStr = ""
			const nextBitsPosition = _iterator.remainingBits - _iterator.offset;
			if (nextBitsPosition >= 0) {
				while (_iterator.remainingBits > nextBitsPosition) {
					let bit = ((_iterator.byteArray[_iterator.byteIndex] >> (7 - _iterator.bitIndex++)) & 1);
					bitsStr += bit;
					value = (value << 1) | bit;
					_iterator.remainingBits--;

					if (_iterator.bitIndex === 8) {
						_iterator.byteIndex++;
						_iterator.bitIndex = 0;
					}
				}
				// Signed numbers specifics
				if (isSigned && _iterator.offset < 32) {
					// Check bit sign based on offset
					const isNegative = (value & (2 ** (_iterator.offset - 1))) > 0;
					if (isNegative) {
						let bitOffset = _iterator.offset;
						value = value | (~0 << bitOffset);
					}
				}
				iterator.current = value;
			} else {
				iterator.current = null; // End of the iterator
			}
			if (currentBitsPerElement != null) { // Go back to configured offset
				_iterator.offset = currentBitsPerElement;
			}
			DebugLog(`\t${bitsStr}`);
			return iterator.current;
		},
		SetBitsPerElement: (bitsPerElement) => {
			if (typeof bitsPerElement !== 'number') {
				throw new Error('bitsPerElement is not a number');
			}
			if (bitsPerElement <= 0) {
				throw new Error('bitsPerElement is not a positive number');
			}
			_iterator.offset = bitsPerElement;
		},
		GetRemainingBits: () => {
			return _iterator.remainingBits;
		},
		Reset: () => {
			_iterator.byteIndex = -1;
			_iterator.bitIndex = 0;
			_iterator.offset = 1;
		}
	};
	return iterator;
};

/**
 * Decode LoRaWAN frame from received payload
 * @param {int[]} byteArray Every elements is a byte (8 bits) with a value from 0 to 255.
 * @param {Date} recvTime Date representing reception of this payload.
 * @returns {Object} Output for the decodeUplink() method with decoded LoRaWAN frame.
 */
function DecodeLorawanFrame(byteArray, recvTime) {
	function SetHeaderFromBitsIterator(bitsIterator) {
		DebugLog("Header : ")
		if (bitsIterator.GetRemainingBits() < 2) {
			_lorawanOutput.errors.push("Not enough bits received (checking id size), payload might be corrupted.");
			return false;
		}
		_lorawanFrame.header.id = bitsIterator.GetNext(2);
		if (!(_lorawanFrame.header.id === 2 || _lorawanFrame.header.id === 3)) {
			_lorawanOutput.errors.push(`Received LoRaWAN frame isn't synchronous nor asynchronous (id=${_lorawanFrame.header.id}), payload might be corrupted or is using information message mode that isn't supported in this codec.`);
			return false;
		}
		_lorawanFrame.bitSizes.header = _lorawanFrame.isAsync ? 55 : 60;
		let decodedHeaderSize = 2;
		if (bitsIterator.GetRemainingBits() < (_lorawanFrame.bitSizes.header - decodedHeaderSize)) {
			_lorawanOutput.errors.push("Not enough bits received (checking header size), payload might be corrupted.");
			return false;
		}
		_lorawanFrame.header.size = bitsIterator.GetNext(11); // entire bit size (header + statementdescriptors + statements) without version field (which is 4 bits)
		decodedHeaderSize += 11;
		if (bitsIterator.length < _lorawanFrame.header.size + 4) {
			_lorawanOutput.errors.push("Not enough bits received (checking specified size and received payload size), payload might be corrupted.");
			return false;
		}
		_lorawanFrame.header.time = new Date(bitsIterator.GetNext(32) * 1000);
		decodedHeaderSize += 32;
		_lorawanFrame.header.offset = bitsIterator.GetNext(7, true) * 15;
		decodedHeaderSize += 7;
		_lorawanOutput.data.TZMinutesOffset = _lorawanFrame.header.offset;
		_lorawanFrame.header.deviceId.sizeId = bitsIterator.GetNext(1);
		decodedHeaderSize += 1;
		if (_lorawanFrame.header.deviceId.sizeId === 1) {
			_lorawanFrame.bitSizes.header += 2;
			if (bitsIterator.GetRemainingBits() < (_lorawanFrame.bitSizes.header - decodedHeaderSize)) {
				_lorawanOutput.errors.push("Not enough bits received (checking header size before decoding deviceId value), payload might be corrupted.");
				return false;
			}
			_lorawanFrame.header.deviceId.value = bitsIterator.GetNext(2) + 1;
			decodedHeaderSize += 2;
		} else {
			_lorawanFrame.header.deviceId.value = 0;
		}
		_lorawanOutput.data.deviceId = _lorawanFrame.header.deviceId.value;
		_lorawanFrame.header.dataCnt.sizeId = bitsIterator.GetNext(2);
		decodedHeaderSize += 2;
		if (_lorawanFrame.header.dataCnt.sizeId > 0) {
			_lorawanFrame.bitSizes.header += _lorawanFrame.header.dataCnt.sizeId;
			if (bitsIterator.GetRemainingBits() < _lorawanFrame.bitSizes.header - decodedHeaderSize) {
				_lorawanOutput.errors.push("Not enough bits received (checking header size before decoding dataCnt value), payload might be corrupted.");
				return false;
			}
			_lorawanFrame.header.dataCnt.value = bitsIterator.GetNext(_lorawanFrame.header.dataCnt.sizeId) + (2 ** _lorawanFrame.header.dataCnt.sizeId);
			decodedHeaderSize += _lorawanFrame.header.dataCnt.sizeId;
		} else {
			_lorawanFrame.header.dataCnt.value = 1;
		}
		if (_lorawanFrame.isSync) {
			_lorawanFrame.header.interval.id = bitsIterator.GetNext(5);
			_lorawanFrame.header.interval.value = SECONDS_INTERVAL_TABLE[_lorawanFrame.header.interval.id];
			if (_lorawanFrame.header.interval.value == null) {
				_lorawanOutput.errors.push(`Can't determine synchronous data interval for interval.id=${_lorawanFrame.header.interval.id}`);
				return false;
			}
		}
		return true;
	};
	function SetStatementDescriptorFromBitsIterator(bitsIterator) {
		let statementDescriptor;
		for (let i = 0; i < _lorawanFrame.header.dataCnt.value; i++) {
			DebugLog("Next statement descriptor : ")
			statementDescriptor = {
				dataId: undefined,
				channel: {
					sizeId: undefined,
					value: undefined
				},
				bitSize: {
					sizeId: undefined,
					value: undefined
				}
			};
			statementDescriptor.dataId = bitsIterator.GetNext(6);
			statementDescriptor.channel.sizeId = bitsIterator.GetNext(2);
			if (statementDescriptor.channel.sizeId > 0 && statementDescriptor.channel.sizeId <= 2) {
				const channelValueBits = statementDescriptor.channel.sizeId === 1 ? 3 : 8;
				const channelConvert = statementDescriptor.channel.sizeId === 1 ? 1 : 9;
				statementDescriptor.channel.value = bitsIterator.GetNext(channelValueBits) + channelConvert;
				_lorawanFrame.bitSizes.statementDescriptors += (12 + channelValueBits);
			} else {
				statementDescriptor.channel.value = 0;
				_lorawanFrame.bitSizes.statementDescriptors += 12;
			}
			statementDescriptor.bitSize.sizeId = bitsIterator.GetNext(4);
			statementDescriptor.bitSize.value = BIT_SIZE_TABLE[statementDescriptor.bitSize.sizeId];
			if (statementDescriptor.bitSize.value == null) {
				_lorawanOutput.errors.push(`Can't determine statement size, statementDescriptor.bitSize.sizeId=${statementDescriptor.bitSize.sizeId}`);
				return false;
			}
			 _lorawanFrame.statementDescriptors.oneSyncStatementBitSize += statementDescriptor.bitSize.value;
			_lorawanFrame.statementDescriptors.value.push(statementDescriptor);
			// Add this statement descriptor to lorawan output
			const datatype = DATA_TYPES[statementDescriptor.dataId];
			if (datatype == null) {
				_lorawanOutput.errors.push(`datatype id not found, id=${statementDescriptor.dataId}`);
				return false;
			}
			const localizedLabel = (LANG != null && typeof LANG === "string") ? datatype[`label${LANG.toUpperCase()}`] : null;
			_lorawanOutput.data.descriptors.push({
				datatype: statementDescriptor.dataId,
				channel: statementDescriptor.channel.value,
				label: localizedLabel != null ? localizedLabel : datatype.label,
				unit: datatype.unit
			})
		}
		return true;
	};
	function AddAllSyncRecords(bitsIterator) {
		DebugLog("Adding all synchronous data");
		const statementsCount = Math.floor(_lorawanFrame.bitSizes.allStatementsBitSize / _lorawanFrame.statementDescriptors.oneSyncStatementBitSize);
		for (let i = 0; i < statementsCount; i++) {
			DebugLog("Statement n°" + (i + 1));
			const statementDate = new Date(_lorawanFrame.header.time.getTime() + (_lorawanFrame.header.interval.value * 1000 * i));
			const record = _lorawanOutput.data.records[statementDate.toISOString()] = {};
			_lorawanFrame.statementDescriptors.value.forEach(function (statementDescriptor, index) {
				const dataType = DATA_TYPES[statementDescriptor.dataId];
				record[index] = dataType.convertFn(bitsIterator.GetNext(statementDescriptor.bitSize.value, dataType.isSigned));
			})
		}
		return true;
	};
	function AddAllAsyncRecords(bitsIterator) {
		DebugLog("Adding all asynchronous data");
		let currentStatementDate = _lorawanFrame.header.time;
		let statementCount = 1;
		while (bitsIterator.GetRemainingBits() > (_lorawanFrame.statementDescriptors.oneSyncStatementBitSize + MIN_ASYNC_SIZE)) {
			DebugLog("Statement n°" + statementCount);
			const sizeId = bitsIterator.GetNext(2);
			const tableInterval = ASYNC_INTERVAL_TABLE[sizeId];
			if (tableInterval == null) {
				_lorawanOutput.errors.push(`Can't determine async interval size for interval sizeId=${sizeId}`);
				return false;
			}
			if (bitsIterator.GetRemainingBits() < (_lorawanFrame.statementDescriptors.oneSyncStatementBitSize + tableInterval.size)) break;
			const interval = bitsIterator.GetNext(tableInterval.size);
			currentStatementDate = new Date(currentStatementDate.getTime() + (tableInterval.convertFn(interval) * 1000));
			// Work-around to keep track of multiples events happening at the same time in right order
			let statementDate = currentStatementDate;
			while (_lorawanOutput.data.records[statementDate.toISOString()]) {
				statementDate = new Date(statementDate.getTime() + 1);
			}
			const record = _lorawanOutput.data.records[statementDate.toISOString()] = {};
			_lorawanFrame.statementDescriptors.value.forEach(function (statementDescriptor, index) {
				const dataType = DATA_TYPES[statementDescriptor.dataId];
				record[index] = dataType.convertFn(bitsIterator.GetNext(statementDescriptor.bitSize.value, dataType.isSigned));
			})
			statementCount++;
		}
		return true;
	};
	function AddRecordsFromBitsIterator(bitsIterator) {
		let result = false;

		if (bitsIterator.GetRemainingBits() < _lorawanFrame.bitSizes.allStatementsBitSize) {
			_lorawanOutput.errors.push("Not enough bits received, payload might be corrupted.");
			return;
		}
		
		DebugLog("Statements : ");
		if (_lorawanFrame.isSync) {
			result = AddAllSyncRecords(bitsIterator);
		} else if (_lorawanFrame.isAsync) {
			result = AddAllAsyncRecords(bitsIterator);
		}

		if (bitsIterator.GetRemainingBits() > 0) {
			const lastBitsSize = bitsIterator.GetRemainingBits();
			const lastBits = bitsIterator.GetNext(lastBitsSize).toString(2).padStart(lastBitsSize, 0);
			DebugLog("Dummy bits : " + lastBits);
		}

		if (_lorawanOutput.data.records.length === 0) {
			_lorawanOutput.warnings.push("No data found");
		}

		return result;
	};
	const _lorawanOutput = {
		data: {
			deviceId: undefined,
			TZMinutesOffset: undefined,
			descriptors: [],
			records: {}
		},
		warnings: [],
		errors: []
	};
	const _lorawanFrame = {
		version: undefined,
		header: {
			id: undefined,
			size: 0,
			time: 0,
			offset: 0,
			deviceId: {
				sizeId: undefined,
				value: undefined
			},
			dataCnt: {
				sizeId: undefined,
				value: undefined
			},
			interval: {
				id: undefined,
				value: undefined
			}
		},
		statementDescriptors: {
			value: [],
			oneSyncStatementBitSize: 0
		},
		bitSizes: {
			header: 0,
			statementDescriptors: 0,
			get allStatementsBitSize() {
				return _lorawanFrame.header.size - _lorawanFrame.bitSizes.header - _lorawanFrame.bitSizes.statementDescriptors;
			}
		},
		get isAsync () {
			return _lorawanFrame.header.id === 3;
		},
		get isSync () {
			return _lorawanFrame.header.id === 2;
		}
	}
	if (recvTime == null || !(recvTime instanceof Date)) {
		DebugLog("no recvTime, using new Date()");
		recvTime = new Date();
	}
	const bitsIterator = GetBitsIterator(byteArray);
	if (bitsIterator == null) {
		_lorawanOutput.errors.push("Bad payload, might be corrupted can't decode.");
		return _lorawanOutput;
	}
	_lorawanFrame.version = bitsIterator.GetNext(4);
	if (_lorawanFrame.version !== 2) {
		_lorawanOutput.errors.unshift(`Unsupported LoRaWAN frame structure version=${_lorawanFrame.version}, can't decode.`);
		return _lorawanOutput;
	}
	if (!SetHeaderFromBitsIterator(bitsIterator)) {
		_lorawanOutput.errors.unshift("Can't decode header.");
		return _lorawanOutput;
	}
	if (!SetStatementDescriptorFromBitsIterator(bitsIterator)) {
		_lorawanOutput.errors.unshift("Can't decode statement descriptors.");
		return _lorawanOutput;
	}
	if (!AddRecordsFromBitsIterator(bitsIterator)) {
		_lorawanOutput.errors.unshift("Can't decode statements.");
	}
	return _lorawanOutput;
}

function decodeUplink(input) {
	try {
		const output = DecodeLorawanFrame(input.bytes, input.recvTime);
		if (output.errors.length > 0 && input.bytes != null && input.bytes.length > 0) {
			const now = new Date();
			const hexPayload = input.bytes
			.map(byte => ('0' + byte.toString(16)).slice(-2))
			.join('');
			output.errors.push(`[${now.toISOString()}] - Received payload = ${hexPayload}`);
		}
		return output;
	} catch (error) {
		return {
            errors: [ error.message ],
            warnings: [],
        };
	}
}